/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/08/07 11:15p $
 * @brief   NM1240 GDMA I2C EEPROM 24LC256 sample code
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

#define CTL_CODE 0xAul
#define EEPROM_SIZE 0x80000 // 32K * 8 (256 Kbit)
#define PAGE_SIZE 0x40 // 64 bytes
#define PAGE_AMOUNT 512
#define PAGE_AMOUNT_Pos 6

volatile uint8_t g_u8EndFlagTX, g_u8EndFlagRX;
uint16_t CTL_BYTE, CHIP_SEL;
uint8_t g_u8BusyFlag;

/* TX data structure */
typedef struct
{
    uint16_t Control_byte;
    uint16_t Address_High_byte;
    uint16_t Address_Low_byte;
    uint16_t data[PAGE_SIZE];
} I2C_TX_T;

/* RX data structure */
typedef struct
{
    uint16_t Control_byte;
    uint16_t data[PAGE_SIZE];
} I2C_RX_T;

I2C_TX_T I2C_TX;
I2C_RX_T I2C_RX;

uint16_t* I2C_RX_ptr;

void Parameter_Init(void)
{
    uint16_t i;
    g_u8EndFlagTX = 0;
    g_u8EndFlagRX = 0;
    CHIP_SEL = 0x2;
    CTL_BYTE = (CTL_CODE << 3) | CHIP_SEL;
    for(i=0;i<PAGE_SIZE;i++)I2C_RX.data[i] = 0;
}

void GDMA0_IRQHandler (void)
{
    /* Disable TIMEOUT INT */
    UI2C_DisableTimeout(UI2C1);
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA0);
    /* Set GDMA0_I2C_RX Flag */
    g_u8EndFlagRX = 1;
    /* Disable GDMA0 INT */
    GDMA_DisableINT(GDMA0);
}

void GDMA1_IRQHandler (void)
{
    /* Disable TIMEOUT INT */
    UI2C_DisableTimeout(UI2C1);
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA1);
    /* Set GDMA1_I2C_TX Flag */
    g_u8EndFlagTX = 1;
    /* Disable GDMA1 INT */
    GDMA_DisableINT(GDMA1);
}

void USCI1_IRQHandler(void)
{
    volatile uint32_t u32Status;
    u32Status = UI2C_GET_PROT_STATUS(UI2C1);

    if (u32Status & UI2C_PROTSTS_ACKIF_Msk) {
        /* Clear USCI_I2C1 ACK Flag */
        UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_ACKIF_Msk);
        /* EEPROM is idle */
        g_u8BusyFlag = 0;
    }
    else if(u32Status & UI2C_PROTSTS_NACKIF_Msk) {
        /* Clear USCI_I2C1 NACK Flag */
        UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_NACKIF_Msk);
        /* EEPROM is busy */
        g_u8BusyFlag = 1;
    }
    else if (u32Status & UI2C_PROTSTS_TOIF_Msk) {
        /* Clear USCI_I2C1 Timeout Flag */
        UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_TOIF_Msk);
        /* Close UI2C */
        UI2C_Close(UI2C1);
    }
}
void SYS_Init(void)
{    
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init System Clock                                                                                       */
    /*---------------------------------------------------------------------------------------------------------*/
    /* Enable HIRC clock */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Waiting for HIRC clock ready */
    CLK_WaitClockReady(CLK_STATUS_HIRCSTB_Msk);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate PllClock, SystemCoreClock and CycylesPerUs automatically. */
    SystemCoreClockUpdate();
}

void UART2_Init(void)
{
    /* Enable USCI2 CLK */
    CLK_EnableModuleClock(USCI2_MODULE);

    /* Reset IP USCI2 */
    SYS_ResetModule(USCI2_RST);

    /* Open USCI2: 115200 */
    UUART_Open(UUART2, 115200);

    /* Set USCI_SPI1 multi-function pins */
    SYS->GPF_MFP &= ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk);
    SYS->GPF_MFP |= ( SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPIO Mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);
}

void I2C1_Init(void)
{
    /* Enable USCI1 CLK */
    CLK_EnableModuleClock(USCI1_MODULE);

    /* Reset IP USCI1 */
    SYS_ResetModule(USCI1_RST);
    
    /* Open UI2C1: 1 MHz */
    UI2C_Open(UI2C1, 1000000);

    /* Asser ACK control. */
    UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_AA);

    /* Enable UI2C ACK, NACK, TIMEROUT INT*/
    UI2C_EnableInt(UI2C1, UI2C_ACK_INT_MASK | UI2C_NACK_INT_MASK | UI2C_TO_INT_MASK);
    /* Enable UI2C1 External Interrupt */
    NVIC_EnableIRQ(USCI1_IRQn);

    /* Set USCI_I2C1 multi-function pins */
    SYS->GPC_MFP &= ~(SYS_GPC_MFP_PC1MFP_Msk | SYS_GPC_MFP_PC2MFP_Msk);
    SYS->GPC_MFP |= (SYS_GPC_MFP_PC0_I2C1_SCL | SYS_GPC_MFP_PC2_I2C1_SDA);
}

void CHECK_EEPROM_IS_IDLE(UI2C_T *ui2c)
{
    uint16_t ctl_data;
    /* Set 24LC256 Control Byte */
    ctl_data = CTL_BYTE<<1 | UI2C_TXDAT_DMASTA_Msk | UI2C_TXDAT_DMASTO_Msk;
    do
    {
        /* Check I2C is not full */
        while(UI2C_IS_TX_FULL(UI2C1));

        /* Clear I2C NACK INT FLAG */
        UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_ACKIF_Msk | UI2C_PROTSTS_NACKIF_Msk | UI2C_PROTSTS_STORIF_Msk);

        /* Send Control Byte */
        UI2C_SET_DATA(UI2C1, ctl_data);

        /* Check I2C is STOP */
        while(!UI2C_GetStatus(ui2c, UI2C_PROTSTS_STORIF_Msk));

    } while (g_u8BusyFlag);
}

void EEPROM_WRITE_DATA(uint16_t contorl_byte, uint16_t address_high_byte, uint16_t address_low_byte, uint16_t write_data_length, uint16_t* i2c_tx_ptr)
{
    /* Check EEPROM is idle */
    CHECK_EEPROM_IS_IDLE(UI2C1);

    /* Enable GDMA1 INT */
    GDMA_EnableINT(GDMA1);

    /* Set device address, write command and add start control at the first transmitting data. */
    *i2c_tx_ptr = contorl_byte<<1 | UI2C_TXDAT_DMASTA_Msk;
    *(i2c_tx_ptr + 1) = address_high_byte;
    *(i2c_tx_ptr + 2) = address_low_byte;

    /* Add Stop control at the end of TX data */
    *(i2c_tx_ptr + 2 + write_data_length) |= UI2C_TXDAT_DMASTO_Msk;  

    /* Check I2C is not full */
    while(UI2C_IS_TX_FULL(UI2C1));

    /* Reset UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMARST_Msk);
    /* Set UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMAEN_Msk | UI2C_DMACTL_TXDMAEN_Msk);

    /* Set transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA1, write_data_length + 3);

    /* Set destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA1, &UI2C1->TXDAT);

    /* Set source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA1, i2c_tx_ptr);

    /* Clear I2C ACK, NACK, TIMEOUT INT FLAG */
    UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_ACKIF_Msk | UI2C_PROTSTS_NACKIF_Msk | UI2C_PROTSTS_STORIF_Msk);

    /* Enable GDMA1 */
    GDMA_ENABLE(GDMA1);

    /* Check the transmission of GDMA1_I2C_TX is complete. */
    while(!g_u8EndFlagTX);
    g_u8EndFlagTX = 0;
}

void EEPROM_READ_DATA(uint16_t contorl_byte, uint16_t address_high_byte, uint16_t address_low_byte, uint16_t read_data_length, uint16_t* i2c_rx_ptr)
{
    /* EEPROM_WRITE_DATA(contorl_byte, address_high_byte, address_low_byte, write_data_length, i2c_tx_ptr) */
    EEPROM_WRITE_DATA(CTL_BYTE, address_high_byte, address_low_byte, 0, &I2C_TX.Control_byte);

    /* Enable GDMA0 INT */
    GDMA_EnableINT(GDMA0);

    /* Set device address, read command and add start control at the first transmitting data */
    *i2c_rx_ptr = contorl_byte<<1 | 0x1 | UI2C_TXDAT_DMASTA_Msk;  

    /* Check I2C is not full */
    while(UI2C_IS_TX_FULL(UI2C1));

    /* Set I2C TX DATA */
    UI2C_SET_DATA(UI2C1, *i2c_rx_ptr);

    /* Set GDMA0 transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA0, read_data_length);

    /* Point to the next data */
    i2c_rx_ptr = i2c_rx_ptr + 1;
    
    /* Set GDMA0 destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA0, i2c_rx_ptr);

    /* Set GDMA0 source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA0, &UI2C1->RXDAT);

    /* Clear I2C NACK INT FLAG */
    UI2C_ClearIntFlag(UI2C1, UI2C_NACK_INT_MASK);

    /* Reset UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMARST_Msk);
    /* Set UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMAEN_Msk | UI2C_DMACTL_RXDMAEN_Msk | UI2C_DMACTL_NACKEN_Msk | UI2C_DMACTL_STOPEN_Msk);

    /* Enable GDMA0 */
    GDMA_ENABLE(GDMA0);

    /* Check the transmission of GDMA0_I2C_RX is complete. */
    while(!g_u8EndFlagRX);
    g_u8EndFlagRX = 0;
}

int32_t InputNumber(void)
{
    int8_t cTemp;
    int32_t Num;
    
    Num = 0;  
    while(1)
    {
        cTemp = getchar();
        printf("%c", cTemp);
            
        if (cTemp == 0x0d)  break;  //enter
        else if (cTemp == 'q' || cTemp == 'Q')  //quit
            Num = cTemp;
        else if (cTemp == 0x08)  //backspace
            Num = Num / 10;
        else if(cTemp >= '0' && cTemp <= '9') 
            Num = Num * 10 + (int32_t)(cTemp - 0x30); 
        else if(cTemp == 0x7F)
            Num = Num / 10;
    }
    printf("\n");
    return (Num);
}

int main()
{
    uint32_t i, j, k;
    uint32_t start_address, high_byte, low_byte, page;
    char sel;
  
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_Init();         /* Initial system clock       */
    UART2_Init();       /* Initial UART2              */
    I2C1_Init();        /* Initial I2C1               */
    SYS_LockReg();      /* Lock protected registers   */

    printf("\n\n");
    printf("+------------------------------------------------+\n");
    printf("|      NM1240 Sample Code for GDMA               |\n");
    printf("+------------------------------------------------+\n");
    printf("Initial NM1240 System Clock:\n");
    printf("CPU clock %dMHz\n",  CLK_GetCPUFreq()/1000000);
    printf("HCLK clock %dMHz\n", CLK_GetHCLKFreq()/1000000);  
    printf("EXT clock %dMHz (0MHz means no clock source)\n", CLK_GetEXTFreq()/1000000);
    printf("Reset status: 0x%08X\n", SYS->RSTSTS);  
    printf("\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    I2C Configuration                     |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| Mode           : Master                                  |\n");
    printf("| Transfer width : 8 bits                                  |\n");
    printf("| Baud Rate      : 1 MHz                                   |\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    GDMA Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| MODE           : USCI to Memory(I2C)                     |\n");
    printf("| Transfer width : 16BITS                                  |\n");
    printf("| Data           : Master write and read 100 data          |\n");
    printf("| Burst mode     : DISABLE                                 |\n");
    printf("+----------------------------------------------------------+\n");
    while(1) {
    printf("                                                            \n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    EEPROM item menu                      |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| 0: Erase data of all address                             |\n");
    printf("| 1: Byte Write                                            |\n");
    printf("| 2: Byte Read                                             |\n");
    printf("| 3: Page Write                                            |\n");
    printf("| 4: Page Read                                             |\n");
    printf("+----------------------------------------------------------+\n");
    printf("Before starting the data transfer, make sure the EEPROM device is ready.\n");
    printf("Select Item");
    sel = getchar();
    printf(": %c\n", sel);

    /* Initial Parameter */
    Parameter_Init();

    /* Reset GDMA */
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_ResetGDMA();
    SYS_LockReg();      /* Lock protected registers */

    /* Open GDMA0 for I2C RX */
    GDMA_OPEN(GDMA0, GDMA_USCI_MODE, GDMA_TWS_16BITS, GDMA_BURST_DISABLE, SOURCE_ADDRESS_FIXED | Destination_ADDRESS_INC);
    /* Enable GDMA0 External Interrupt */
    NVIC_EnableIRQ(GDMA0_IRQn);
    
    /* Open GDMA1 for I2C TX */
    GDMA_OPEN(GDMA1, GDMA_USCI_MODE, GDMA_TWS_16BITS, GDMA_BURST_DISABLE, SOURCE_ADDRESS_INC | Destination_ADDRESS_FIXED);
    /* Enable GDMA1 External Interrupt */
    NVIC_EnableIRQ(GDMA1_IRQn);
    
    switch (sel)
    {
    case '0': //Erase data of all address
        for(j=0;j<PAGE_AMOUNT;j++) 
        {
            /* Set data of transmission */
            for(i=0;i<PAGE_SIZE;i++)I2C_TX.data[i] = 0xFF;
            
            /* Set Write address */
            start_address = (j<<PAGE_AMOUNT_Pos);
            low_byte = start_address & 0xFF;
            high_byte = (start_address>>8) & 0x7F;
            printf("\rErasing...");
            
            /* EEPROM_WRITE_DATA(contorl_byte, address_high_byte, address_low_byte, write_data_length, i2c_tx_ptr) */
            EEPROM_WRITE_DATA(CTL_BYTE, high_byte, low_byte, PAGE_SIZE, &I2C_TX.Control_byte);
        }
        printf("...Compete!\n");
    break;
    
    case '1': //Byte Write
        printf("Write address(0~32767): ");
        start_address = InputNumber();
        
        /* Set Write address */
        if (start_address > (EEPROM_SIZE-1)) start_address = EEPROM_SIZE-1;
        low_byte = start_address & 0xFF;
        high_byte = (start_address>>8) & 0x7F;
        page = start_address >> PAGE_AMOUNT_Pos;
        printf("EEPROM page: %d, address: %d\n", page, start_address);
        printf("Write data(0~255): ");
        I2C_TX.data[0] = InputNumber();
        if (I2C_TX.data[0] > 255) I2C_TX.data[0] = 255;
        
        /* EEPROM_WRITE_DATA(contorl_byte, address_high_byte, address_low_byte, write_data_length, i2c_tx_ptr) */
        EEPROM_WRITE_DATA(CTL_BYTE, high_byte, low_byte, 1, &I2C_TX.Control_byte);
    break;
    
    case '2': //Byte Read
        printf("Read address(0~32767): ");
        start_address = InputNumber();
        
        /* Set Read address */
        if (start_address > (EEPROM_SIZE-1)) start_address = EEPROM_SIZE-1;
        low_byte = start_address & 0xFF;
        high_byte = (start_address>>8) & 0x7F;
        page = start_address >> PAGE_AMOUNT_Pos;
        printf("EEPROM page: %d, address: %d\n", page, start_address);
    
        /* EEPROM_READ_DATA(contorl_byte, address_high_byte, address_low_byte, read_data_length, i2c_rx_ptr) */
        EEPROM_READ_DATA(CTL_BYTE, high_byte, low_byte, 1, &I2C_RX.Control_byte);
        printf("EEPROM[%d] = %d (0x%02X)\n", start_address, I2C_RX.data[0], I2C_RX.data[0]);
    break;
    
    case '3': //Page Write
        printf("Write Page(0~511): ");
        page = InputNumber();
        
        /* Set Write address */
        if (page > (PAGE_AMOUNT-1)) page  = (PAGE_AMOUNT-1);
        start_address = (page<<PAGE_AMOUNT_Pos);
        low_byte = start_address & 0xFF;
        high_byte = (start_address>>8) & 0x7F;
        for(i=0;i<PAGE_SIZE;i++) I2C_TX.data[i] = i;
        
        /* EEPROM_WRITE_DATA(contorl_byte, address_high_byte, address_low_byte, write_data_length, i2c_tx_ptr) */
        EEPROM_WRITE_DATA(CTL_BYTE, high_byte, low_byte, PAGE_SIZE, &I2C_TX.Control_byte);        
    break;
    
    case '4': //Page Read
        printf("Read Page(0~511): ");
        page = InputNumber();
        
        /* Set Read address */
        if (page > (PAGE_AMOUNT-1)) page  = (PAGE_AMOUNT-1);
        start_address = (page<<PAGE_AMOUNT_Pos);
        low_byte = start_address & 0xFF;
        high_byte = (start_address>>8) & 0x7F;
        
        /* EEPROM_READ_DATA(contorl_byte, address_high_byte, address_low_byte, write_data_length, i2c_rx_ptr) */
        EEPROM_READ_DATA(CTL_BYTE, high_byte, low_byte, PAGE_SIZE, &I2C_RX.Control_byte);
        k = 0;
        for(i=(page<<PAGE_AMOUNT_Pos);i<((page+1)<<PAGE_AMOUNT_Pos);i++) {
            printf("EEPROM[%d] = %d (0x%02X)\n", i, I2C_RX.data[k], I2C_RX.data[k]);
            k++;
        }
    break;
    
    default:
        printf("Error select!\n");
    break;
    }
  }  // end of while(1)
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
